import os
import subprocess
import sys

# ========= CONFIGURAÇÕES =========
VIDEO_DIR = "/var/project/videos/drige"
CUT_DIR = "/var/project/videos/cortes"
EXTS = (".mp4", ".mkv", ".avi", ".mov")

# ========= FUNÇÕES =========

def clear():
    os.system("clear")

def listar_pastas():
    return [d for d in os.listdir(VIDEO_DIR)
            if os.path.isdir(os.path.join(VIDEO_DIR, d))]

def listar_videos(pasta):
    path = os.path.join(VIDEO_DIR, pasta)
    return [f for f in os.listdir(path) if f.lower().endswith(EXTS)]

def escolher(lista, titulo):
    print(f"\n📁 {titulo}")
    for i, item in enumerate(lista):
        print(f"[{i}] {item}")
    try:
        return lista[int(input("\nEscolha o número: "))]
    except:
        print("❌ Opção inválida")
        sys.exit(1)

def tempo_para_segundos(tempo):
    try:
        if ":" in tempo:
            m, s = tempo.split(":")
            return int(m) * 60 + int(s)
        return int(tempo)
    except:
        raise ValueError("Formato inválido")

def segundos_para_tempo(seg):
    return f"{seg//60:02d}:{seg%60:02d}"

def cortar_video(src, start, duration, output):
    os.makedirs(os.path.dirname(output), exist_ok=True)

    cmd = [
        "ffmpeg", "-y",
        "-ss", str(start),
        "-t", str(duration),
        "-i", src,

        "-c:v", "libx264",
        "-preset", "ultrafast",
        "-crf", "23",
        "-profile:v", "high",
        "-level", "4.1",
        "-pix_fmt", "yuv420p",
        "-tune", "fastdecode",

        "-g", "15",
        "-keyint_min", "15",
        "-sc_threshold", "0",
        "-bf", "0",
        "-refs", "1",

        "-c:a", "aac",
        "-b:a", "128k",
        "-ar", "48000",

        "-movflags", "+faststart",
        output
    ]

    subprocess.run(cmd)

# ========= MAIN =========

def main():
    clear()
    print("✂️ CUTTER CLI — Modo Profissional\n")

    pasta = escolher(listar_pastas(), "Pastas disponíveis")
    video = escolher(listar_videos(pasta), "Vídeos disponíveis")

    src = os.path.join(VIDEO_DIR, pasta, video)

    clear()
    print(f"🎬 Vídeo selecionado: {video}")

    while True:
        try:
            start_input = input("\n⏱️ Tempo inicial (MM:SS ou segundos): ")
            duration_input = input("⏱️ Duração do corte (MM:SS ou segundos): ")

            start = tempo_para_segundos(start_input)
            duration = tempo_para_segundos(duration_input)
        except:
            print("❌ Formato inválido. Use MM:SS ou segundos")
            continue

        nome = input("📄 Nome do arquivo (Enter = automático): ").strip()
        if not nome:
            nome = f"cut_{segundos_para_tempo(start)}_{segundos_para_tempo(duration)}"

        output = os.path.join(CUT_DIR, f"{nome}.mp4")

        print("\n📌 RESUMO")
        print(f"▶️ Início: {segundos_para_tempo(start)}")
        print(f"⏹️ Duração: {segundos_para_tempo(duration)}")
        print(f"💾 Saída: {output}")

        conf = input("\n[C] cortar | [Q] sair | Enter cancelar: ").lower()

        if conf == "q":
            print("👋 Saindo...")
            break
        elif conf != "c":
            print("↩️ Cancelado, voltando ao vídeo...")
            continue

        cortar_video(src, start, duration, output)

        print("\n✅ Corte finalizado!")
        print("➡️ Você continua no MESMO vídeo")

# ========= RUN =========
if __name__ == "__main__":
    main()
